//**********************************************************************************************
//		(C) Copyright 2003 by Dipl. Phys. Joerg Plewe, HARDCODE Development
//		All rights reserved. Copying, modification,
//		distribution or publication without the prior written
//		consent of the author is prohibited.
//**********************************************************************************************
package de.jpilot.game;

import java.awt.Dimension;

/**
 *
 * @author  Herkules
 */
public class Ship extends Actor
{	

	private			boolean				mIsDead			= true;
	private final	PlayerStatistics	mStatistics		= new PlayerStatistics( this );

	/** Each ship has a name for the GUI. */
	String mName = null;

	
	/**
	 * Notifier that notifies about dying.
	 */
	private final Notifier DIED = new Notifier()
	{
		public void notify( ActorListener listener )
		{
			if ( listener instanceof ShipListener )
				((ShipListener)listener).died( Ship.this );
		}
	};

	
	/**
	 * Notifier that notifies about spawning.
	 */
	private final Notifier SPAWNED = new Notifier()
	{
		public void notify( ActorListener listener )
		{
			if ( listener instanceof ShipListener )
				((ShipListener)listener).spawned( Ship.this );
		}
	};
	
	/**
	 * Notifier that notifies about closing.
	 */
	class KilledNotifier implements Notifier
	{
		private Ship mWhom;
		
		void set( Ship whom )
		{
			mWhom = whom;
		}
		public void notify( ActorListener listener )
		{
			if ( listener instanceof ShipListener )
				((ShipListener)listener).killed( Ship.this, mWhom );
		}
	};
	private final KilledNotifier KILLED = new KilledNotifier();
	
	/**
	 * Notifier that notifies about renaming.
	 */
	private final Notifier RENAMED = new Notifier()
	{
		public void notify( ActorListener listener )
		{
			if ( listener instanceof ShipListener )
				((ShipListener)listener).renamed( Ship.this );
		}
	};

	
	/**
	 * Notifier that notifies about damage.
	 */
	private class DamagedNotifier implements Notifier
	{
		private float	mHowMuch;
		private	Ship	mWho;
		
		/** Allow to configure this single instance. */
		void set( float howmuch, Ship who )
		{
			mHowMuch	= howmuch;
			mWho		= who;
		}
		
		public void notify( ActorListener listener )
		{
			if ( listener instanceof ShipListener )
				((ShipListener)listener).damaged( Ship.this, mHowMuch, mWho );
		}
	};
	private final DamagedNotifier DAMAGED = new DamagedNotifier();
	
	
	/**
	 * Creates a new instance of Player
	 */
	public Ship( Game game )
	{
		super( game );
	}	
			
	
	/**
	 * Deliver statistics.
	 */
	public PlayerStatistics getStatistics()
	{
		return mStatistics;
	}

	
	/**
	 * Go dead.
	 */
	public void die()
	{
		mIsDead = true;

		// Maintain my stats.
		getStatistics().addKilled();

		notify( DIED );
	}
	
	
	/**
	 * Go alive.
	 */
	public void spawn()
	{
		mIsDead = false;

		// Maintain my stats.
		getStatistics().setBounty( 0 );

		notify( SPAWNED );
	}
	
	
	/**
	 * Deliver live status.
	 */
	public boolean isDead()
	{
		return mIsDead;
	}

	
	/**
	 * Notification that this Ship killed somebody.
	 */
	public void killed( Ship whom )
	{
		// Maintain my stats.
		getStatistics().addKills();
		getStatistics().addBounty( whom.getStatistics().getBounty() + 1 );
		
		KILLED.set( whom );
		notify( KILLED );
	}
	
	
	/** 
	 * Getter for property Name.
	 * @return Value of property Name.
	 */
	public String getName()
	{
		return mName;
	}
	
	/** 
	 * Setter for property Name.
	 * @param name New value of property Name.
	 */
	public void setName( String name )
	{
		this.mName = name;
		notify( RENAMED );
	}
	
	
	/**
	 * Apply damage to the ship.
	 */
	public void damage( float howmuch, Ship who )
	{
		DAMAGED.set( howmuch, who );
		notify( DAMAGED );
	}
	

	/**
	 * Helper: 
	 * Place at a random position.
	 */
	public void placeRandom()
	{
		Dimension mapsize = getGame().getMap().getSize();
		MotionData data = this.getMotionData();
		
		data.x		= (float)(Math.random() * mapsize.getWidth() );
		data.y		= (float)(Math.random() * mapsize.getHeight() );
		data.vx		= 0.0f;
		data.vy		= 0.0f;
		data.ax		= 0.0f;
		data.ay		= 0.0f;
		data.alpha	= (float)(Math.random() * 2.0 * Math.PI );
		data.omega	= 0.0f;
		this.setMotionData( data );
	}
	
}


