//**********************************************************************************************
//		(C) Copyright 2003 by Dipl. Phys. Joerg Plewe, HARDCODE Development
//		All rights reserved. Copying, modification,
//		distribution or publication without the prior written
//		consent of the author is prohibited.
//**********************************************************************************************
package de.jpilot.game;

import de.hardcode.time.Ticker;
import de.jpilot.commands.Command;
import de.jpilot.commands.CommandQueue;
import java.util.ArrayList;
import java.util.Iterator;

/**
 *
 * @author  Herkules
 */
public class Game
{
	private final	Map				mMap;
	private final	CommandQueue	mCommands		= new CommandQueue();
	private final	ArrayList		mActors			= new ArrayList();
	private			Ticker			mTicker			= null;
	
	/**
	 * Creates a new instance of Game
	 */
	public Game( Map map )
	{
		mMap = map;
	}
	
	
	/** 
	 * Shutdown.
	 */
	public synchronized void close()
	{
		
		//
		// Close all open actors.
		// I don't have to close all players as well, bc. every player
		// also is an actor.
		//
		Iterator it = getActors();
		while ( it.hasNext() )
		{
			Actor a = (Actor)it.next();
			a.close();
		}
	}
	
	
	/**
	 * Retrieve the Map.
	 */
	public final Map getMap()
	{
		return mMap;
	}

	/**
	 * Retrieve the CommandQueue.
	 */
	public final CommandQueue getCommandQueue()
	{
		return mCommands;
	}
	
	
	/**
	 * Retrieve ticker.
	 * Only valid during the update cycle!
	 */
	public final Ticker getTicker()
	{
		return mTicker;
	}
	
	
	/**
	 * Keep track of actors.
	 */
	void addActor( Actor actor )
	{
		mCommands.add( new AddActor( actor ) );
	}

	/**
	 * Keep track of actors.
	 */
	void removeActor( Actor actor )
	{
		mCommands.add( new RemoveActor( actor ) );
	}
	

	/** 
	 * Enumerate all actors.
	 */
	Iterator getActors()
	{
		return mActors.iterator();
	}
		
	
	/**
	 * Step the game in time.
	 */
	public synchronized void update( Ticker ticker )
	{
		// Note ticker.
		mTicker = ticker;
		
		//
		// Update all Actors.
		//
		Iterator it = getActors();
		while ( it.hasNext() )
		{
			Actor a = (Actor)it.next();
			a.update( ticker );
		}

		// Evaluate the commandqueue.
		mCommands.update( ticker );
		
		// Invalidate ticker - outside of update() a game has no understanding
		// of time.
		mTicker = null;
	}
	
	
	//
	//
	// Some internal commands
	// 
	//
	
	class RemoveActor extends Command
	{
		private final Actor mActor;
		
		RemoveActor( Actor actor )
		{
			mActor = actor;
		}
		
		protected void execute( Ticker ticker )
		{
			mActors.remove( mActor );
			setState( Command.FINISHED );
		}
	}

	class AddActor extends Command
	{
		private final Actor mActor;
		
		AddActor( Actor actor )
		{
			mActor = actor;
		}
		
		protected void execute( Ticker ticker )
		{
			mActors.add( mActor );
			setState( Command.FINISHED );
		}
	}

	
}
