//**********************************************************************************************
//		(C) Copyright 2003 by Dipl. Phys. Joerg Plewe, HARDCODE Development
//		All rights reserved. Copying, modification,
//		distribution or publication without the prior written
//		consent of the author is prohibited.
//**********************************************************************************************
package de.jpilot.game;

import de.hardcode.hq.identity.*;
import de.hardcode.time.Ticker;
import java.awt.Dimension;
import java.util.ArrayList;

/**
 *
 * @author  Herkules
 */
public class Actor implements GameObject
{

	/** Remember where I belong to. */
	private final Game mGame;
	
	/** Position and speed. */
	protected MotionData mMotionData = new MotionData();
	
	/** Hold a list of handlers that treat me. */
	private final ArrayList mHandlers	= new ArrayList();
	
	/** Hold a list of listeners that observe me. */
	private final ArrayList mListeners	= new ArrayList();

	/**
	 * Base interface for the notification mechanism.
	 */
	interface Notifier
	{
		void notify( ActorListener listener );
	};
	
	
	/**
	 * Notifier that notifies about moving.
	 */
	private final Notifier MOVED = new Notifier()
	{
		public void notify( ActorListener listener )
		{
			listener.moved( Actor.this );
		}
	};
	
	/**
	 * Notifier that notifies about closing.
	 */
	private final Notifier CLOSED = new Notifier()
	{
		public void notify( ActorListener listener )
		{
			listener.closed( Actor.this );
		}
	};
	
	
	
	/**
	 * Create a new instance of Actor.
	 */
	public Actor( Game game )
	{
		mGame = game;

		// Register myself with the game.
		mGame.addActor( this );		
	}
	
	
	/**
	 * Shutdown the actor.
	 */
	public void close()
	{
		// Unregister myself from the game.
		mGame.removeActor( this );
		closeHandlers();
		notify( CLOSED );
	}

	
	/**
	 * Main update.
	 */
	public void update( Ticker ticker )
	{
		updateHandlers( ticker );
	}
	
	
	/** 
	 * Retrieve the game I play in.
	 */
	public Game getGame()
	{
		return mGame;
	}
	
		
	/** 
	 * Set new position.
	 */
	public void setMotionData( MotionData data )
	{
		mMotionData.set( data );
		notify( MOVED );
	}	
	
	
	/** 
	 * Retrieve motiondata.
	 */
	public MotionData getMotionData()
	{
		return new MotionData( mMotionData );
	}

	
	/** 
	 * Retrieve motiondata.
	 */
	public void getMotionData( MotionData data )
	{
		mMotionData.get( data );
	}

	
	/** 
	 * Retrieve a motiondata reference.
	 * Handle with care! Don't write to the object delivered here!
	 */
	public MotionData getMotionDataRef()
	{
		return mMotionData;
	}

		
	/**
	 * Notify about renaming.
	 */
	protected void notify( Notifier not )
	{
		int cnt = mListeners.size();
		for ( int i = cnt-1; i >= 0; i-- )
		{
			ActorListener l = (ActorListener)mListeners.get( i );
			not.notify( l );
		}		
	}
	
	
	/**
	 * Add another handler.
	 */
	public void addHandler( ActorHandler handler )
	{
		mHandlers.add( handler );
	}
	
	
	/**
	 * Remove a handler.
	 */
	public void removeHandler( ActorHandler handler )
	{
		mHandlers.remove( handler );
	}
	
	
	/** 
	 * Helper to evaluate all handlers.
	 */
	protected void updateHandlers( Ticker ticker )
	{
		final int cnt = mHandlers.size();
		for ( int i = 0; i < cnt; i++ )
		{
			ActorHandler h = (ActorHandler)mHandlers.get( i );
			h.update( this, ticker );
		}
	}

	
	/**
	 * Shutdown all handlers.
	 */
	protected void closeHandlers()
	{
		int cnt = mHandlers.size();
		for ( int i = 0; i < cnt; i++ )
		{
			ActorHandler h = (ActorHandler)mHandlers.get( i );
			h.close();
		}
		
		// DON'T clear the handlers! A handler itself may close the Actor during update!
//		mHandlers.clear();
	}

	/**
	 * Add another listener.
	 */
	public void addListener( ActorListener listener )
	{
		mListeners.add( listener );
	}
	
	
	/**
	 * Remove a listener.
	 */
	public void removeListener( ActorListener listener )
	{
		mListeners.remove( listener );
	}
}

