//**********************************************************************************************
//		(C) Copyright 2003 by Dipl. Phys. Joerg Plewe, HARDCODE Development
//		All rights reserved. Copying, modification,
//		distribution or publication without the prior written
//		consent of the author is prohibited.
//**********************************************************************************************
package de.jpilot.app.server;

import de.jpilot.protocol.SubSystems;
import de.hardcode.hq.chat.client.*;
import de.hardcode.hq.chat.server.ChatServer;
import de.hardcode.hq.location.server.LocationsServer;
import de.hardcode.hq.objectbus.BroadcastServerListener;
import de.hardcode.hq.objectbus.NetStation;
import de.hardcode.hq.objectbus.ServerBusStationDriver;
import de.hardcode.hq.property.server.PropertyServerModule;
import de.hardcode.hq.registry.server.RegistryServerModule;
import de.hardcode.hq.time.server.TimeService;
import de.hardcode.time.SystemClock;
import de.jpilot.protocol.ID;
import java.io.*;

/**
 *
 * @author  Herkules
 */
public class Server
{
    public final static int SERVER_DEFAULT_PORT = 13664;
    
    private     NetStation              mStation;
    private     SubSystems              mSubSystems         = new SubSystems();
    private     ChatClient				mChat;
	private		ChatServer				mChatServer;
	private		PropertyServerModule	mPropertyServer;
	private		RegistryServerModule	mRegistryServer;
	private		LocationsServer			mLocationsServer;
	private		TimeService				mTimeService;
	private		BroadcastServerListener	mPublicEvents;
	
	/**
	 * Creates a new instance of Server
	 */
	public Server()
	{
		this( SERVER_DEFAULT_PORT );
	}	
	
	
	/**
	 * Creates a new instance of Server
	 */
	public Server( int port )
	{
		// Redirect stdout and stderr
		setIO();

		// Welcome
		System.out.println( "Welcome to the JPilot server!" );
		
		mStation = new NetStation( port );
		if ( mStation.isOperational() )
		{
			mStation.startReceive();		

			setupChat();
			mChatServer			= new ChatServer( mStation, ID.CHAT );
			mPropertyServer		= new PropertyServerModule( mStation, ID.PROPERTY, ID.PROPERTY_SERVICE, mSubSystems.getProperties() );
			mRegistryServer		= new RegistryServerModule( mStation, ID.REGISTRY, ID.REGISTRY_SERVICE, mSubSystems.getRegistry() );
			mLocationsServer	= new LocationsServer( mStation, ID.LOCATIONS );
			mTimeService		= new TimeService( mStation, ID.TIME_SERVICE, new SystemClock() );

			// Install a simple broadcaster to process public events.
			mPublicEvents = new BroadcastServerListener( mStation, ID.PUBLIC_EVENTS );


			// Last, act as a server.
			new ServerBusStationDriver( mStation );
		}
		else
		{
			// ouch!
	        System.out.println( "Sorry, cannot initialize the server correctly!" );		
		}
	}

	
	/**
	 * Does the server feel ok?
	 */
	public boolean isOperational()
	{
		return mStation.isOperational();
	}
	
    /**
     * Shutdown.
     */
    public void close()
    {
        System.out.println( "Shutting down...good bye!" );
        mChat.close();
		mPropertyServer.close();
		mRegistryServer.close();
		mLocationsServer.close();
		mTimeService.close();
		mPublicEvents.close();
		
        mStation.close();
        mSubSystems.close();
    }

	
	void setupChat()
	{
		//
		// Configure chat
		//
		mChat = new ChatClient( mStation, ID.CHAT );
        mChat.addListener( 
			new ChatListener()
            {   
                public void newMessage( ChatMessage msg )
                {
                    System.out.println( msg.getMessage() );
                    mChat.clear();
                }
            } 
        );
	}
	
	
	/**
	 * Main loop of the server.
	 */
    void operateServerInteractiv()
    {
        try
        {
            for(;;)
            {
                BufferedReader br = new BufferedReader( new InputStreamReader( System.in ) );
                String s = br.readLine();
                
                if ( s.length() > 0 )
                {
                    if ( s.equals( "quit" ) )
                    {
                        this.close();
                        return;
                    }
                    else if ( s.equals( "help" ) )
                    {
                        System.out.println(": Commands are:");
                        System.out.println(": help\t\tGets you here.");
                        System.out.println(": quit\t\tShutdown server and exit.");
                    }
                    else
                    {
                        mChat.send( "<server operator>", s );
                        mStation.flush();
                    }
                }
                
            }
        }
        catch ( IOException ioex )
        {
            System.out.println("Encountered unexpected error: " + ioex );
        }
        
        this.close();
    }

	
	/**
	 * Sets application IO streams to files based on properties.
	 * The three properties are:
	 *		in - System.in
	 *		out - System.out
	 *		err - System.err
	 */
	public void setIO()
	{
		try
		{
			System.setOut(setO( System.out, "out" ));
			System.setErr(setO( System.err, "err" ));
		}
		catch ( IOException ioex )
		{
		}
	}
	
	/**
	 * Insert the method's description here.
	 * Creation date: (15/02/2001 6:19:10 PM)
	 */
	private PrintStream setO(PrintStream output, String fileName) throws FileNotFoundException
	{
		if (fileName == null || fileName.equals(""))
			return output;
		
		boolean append = true;
		File outFile = new File(fileName);
		if (outFile.exists())
		{
			if ( outFile.length() > 65535 )
			{
				outFile.renameTo(new File(fileName + ".old"));
				append = false;
			}
		}
		return new PrintStream(new FileOutputStream(fileName, append))
		{
			public void println(String s)
			{
				print( new java.util.Date() );
				print('\t');
				super.println(s);
			}
		};		
	}
	
	
	
    /**
     * @param args the command line arguments
     */
    public static void main(String args[])
    {
		int port = SERVER_DEFAULT_PORT;
		
        if ( args.length == 1 )
        {
            try
            {
                port = Integer.parseInt( args[ 0 ] );
            }
            catch ( NumberFormatException e )
            {
                System.out.println( "Argument " + args[ 0 ] + " isn't a valid port number!" );
            }
        }
		
        Server server = new Server( port );
		server.operateServerInteractiv();
    }
    	
}
